using System;
using System.Collections;
using System.Data;
using System.Data.SqlClient;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.ExceptionManagement;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using TABLE = gov.va.med.vbecs.Common.VbecsTables.ProductModificationParameter;

namespace gov.va.med.vbecs.BOL
{
		#region Header

		///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/23/2003</CreationDate>
		///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		///<summary>
		///Blood Product Modifications deals with the modifications that can take place onthe blood product i.e. Thaw.
		///</summary>

		#endregion

	public class BloodProductModifications : BaseBusinessObject
	{
		#region Variables

		///	<summary>
		///	Indicates whether or not the modification has been chosen.
		///	</summary>
		private Hashtable _productModifications;
		///	<summary>
		///	Holds the RowVersion for each of the modifications.
		///	</summary>
		private Hashtable _rowVersions;
		/// <summary>
		/// Holds the cost of each modification
		/// </summary>
		private Hashtable _modificationCosts;

		#endregion

		#region Constructors

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/20/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2130"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid BloodProductModifications object</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="594"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Empty constructor
		/// </summary>
		public BloodProductModifications()
		{
			this._modificationCosts = new Hashtable();
			this._productModifications = new Hashtable();
			this._rowVersions = new Hashtable();
		}

		
		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/21/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="959"> 
		///		<ExpectedInput>Valid division code</ExpectedInput>
		///		<ExpectedOutput>Valid BloodProductModifications object</ExpectedOutput>
		///	</Case>
		///		
		///<Case type="1" testid ="2131"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// BloodProductModifications constructor
		/// BR_26.32 - Grabs product modifications for a division.  Allows the PL to select
		/// active from inactive choices via LoadProductModifications() method.
		/// </summary>
		/// <param name="divisionCode"></param>
		public BloodProductModifications(string divisionCode)
		{
			// Populate _productModifications Hashtable with all of the possible product modifications and false.
			//  An enumeration is used because not all of the fields from the database are used.
			this._productModifications = new Hashtable();
			this._productModifications.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.SplitAdult),false);
			this._productModifications.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Deglycerolized),false);
			this._productModifications.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Frozen),false);
			this._productModifications.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Irradiated),false);
			this._productModifications.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.LeukoReduced),false);
			this._productModifications.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Pooled),false);
			this._productModifications.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Rejuvenated),false);
			this._productModifications.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Thawed),false);
			this._productModifications.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.PoolThawCryo),false);
			this._productModifications.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.VolumeReduced),false);
			this._productModifications.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Washed),false);
			//
			#region Split/Divide - Pediatric: Future Use Only
			//
			//	At this time we are only updating adults, so the following may be added in the future:
			//
			//	this._productModifications.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.SplitPediatric),false);
			//
			#endregion

			this._rowVersions = new Hashtable();
			this._rowVersions.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.SplitAdult),new byte[0]);
			this._rowVersions.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Deglycerolized),new byte[0]);
			this._rowVersions.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Frozen),new byte[0]);
			this._rowVersions.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Irradiated),new byte[0]);
			this._rowVersions.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.LeukoReduced),new byte[0]);
			this._rowVersions.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Pooled),new byte[0]);
			this._rowVersions.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Rejuvenated),new byte[0]);
			this._rowVersions.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Thawed),new byte[0]);
			this._rowVersions.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.PoolThawCryo),new byte[0]);
			this._rowVersions.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.VolumeReduced),new byte[0]);
			this._rowVersions.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Washed),new byte[0]);

			this._modificationCosts = new Hashtable();
			this._modificationCosts.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.SplitAdult),(decimal)0.00);
			this._modificationCosts.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Deglycerolized),(decimal)0.00);
			this._modificationCosts.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Frozen),(decimal)0.00);
			this._modificationCosts.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Irradiated),(decimal)0.00);
			this._modificationCosts.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.LeukoReduced),(decimal)0.00);
			this._modificationCosts.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Pooled),(decimal)0.00);
			this._modificationCosts.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Rejuvenated),(decimal)0.00);
			this._modificationCosts.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Thawed),(decimal)0.00);
			this._modificationCosts.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.PoolThawCryo),(decimal)0.00);
			this._modificationCosts.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.VolumeReduced),(decimal)0.00);
			this._modificationCosts.Add(Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Washed),(decimal)0.00);
			//
			this.LoadProductModifications(divisionCode);
		}

		#endregion
		
		#region Properties

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/22/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="961"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Hashtable with product modifications</ExpectedOutput>
		///	</Case>
		///		
		///<Case type="1" testid ="2879"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Hashtable of product modifications
		/// Implemements BR_10.03, TT_26.02
		/// </summary>
		public Hashtable ProductModifications
		{
			get
			{
				return _productModifications;
			}
			set 
			{
				_productModifications = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/22/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2393"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Hashtable with product modifications</ExpectedOutput>
		///	</Case>
		///		
		///<Case type="1" testid ="595">
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Hashtable of product modification costs
		/// Implemements BR_10.03, TT_26.02
		/// </summary>
		public Hashtable ModificationCosts
		{
			set
			{
				_modificationCosts = value;
			}
			get
			{
				return _modificationCosts;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/22/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="6213"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>boolean</ExpectedOutput>
		///	</Case>
		/// 
		///<Case type="1" testid ="7378"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns indication of whether any product modifications are currently enabled
		/// BR_9.10
		/// </summary>
		public bool IsProductModificationEnabled
		{
			get
			{
				IDictionaryEnumerator myEnumerator = this.ProductModifications.GetEnumerator();
				while ( myEnumerator.MoveNext() )
				{
					if ((bool)myEnumerator.Value)
					{
						return true;
					}
				}
				return false;
			}
		}

		#endregion

		#region Methods

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/21/2002</CreationDate>
		///<summary>
		/// This method loads all of the valid status codes for a given division's 
		/// Components.  These Product Modifications are stored in the _productModifications
		/// Hashtable.  The RowVersion for each modification is stored in the _rowVersions
		/// Hashtable. 
		///</summary>
		private void LoadProductModifications(string division)
		{
			if (division == null || division.Trim().Length == 0)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument("division").ResString);
			}

			DataTable dtProductModifications = DAL.BloodProductModifications.GetExistingProductModifications(division);
			
			foreach (DataRow dtRow in dtProductModifications.Rows)
			{
				char key = dtRow[TABLE.ProductModificationCode].ToString()[0];

				if (this._rowVersions.ContainsKey(key))
				{
					this._rowVersions[key] = (byte[]) dtRow[TABLE.RowVersion];
				}
								
				string statusCode = (string) dtRow[Common.VbecsTables.BloodProduct.RecordStatusCode];
				if (Common.RecordStatusCode.Active == Common.Utility.GetRecordStatusCodeFromString(statusCode))
				{
					if (this._productModifications.ContainsKey(key))
					{
						this._productModifications[key] = true;
					}
				}

				if (this._modificationCosts.ContainsKey(key))
				{
					this._modificationCosts[key] = (decimal)dtRow[TABLE.ProductModificationCost];
				}
			}
		}


		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/21/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="960"> 
		///		<ExpectedInput>Valid division code</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2293"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Updates product modifications table
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		public bool UpdateProductModifications(string divisionCode, Common.UpdateFunction lastUpdateFunctionId)
		{
			IDictionaryEnumerator enumerator = _productModifications.GetEnumerator();
			DataTable dtModifications = new DataTable();
			
			dtModifications.Columns.Add(TABLE.ProductModificationParameterGuid, typeof(System.Guid));
			dtModifications.Columns.Add(TABLE.ProductModificationCode, typeof(string));
			dtModifications.Columns.Add(TABLE.ProductModificationCost, typeof(System.Decimal));
			dtModifications.Columns.Add(TABLE.DivisionCode, typeof(string));
			dtModifications.Columns.Add(TABLE.RecordStatusCode, typeof(string));
			dtModifications.Columns.Add(TABLE.LastUpdateUser, typeof(string));
			dtModifications.Columns.Add(TABLE.RowVersion, typeof(byte[]));
			dtModifications.Columns.Add(TABLE.LastUpdateFunctionId, typeof(int));

			while (enumerator.MoveNext()) 
			{
				DataRow drModification = dtModifications.NewRow();
				drModification[TABLE.ProductModificationParameterGuid] = Guid.NewGuid();
				drModification[TABLE.ProductModificationCode] = (char) enumerator.Key;
				drModification[TABLE.ProductModificationCost] = (decimal) _modificationCosts[enumerator.Key];
				drModification[TABLE.DivisionCode] = divisionCode;
				if ((bool) enumerator.Value == true) 
				{
					drModification[TABLE.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
				}
				else
				{
					drModification[TABLE.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive);
				}
				drModification[TABLE.LastUpdateUser] = Common.LogonUser.LogonUserName;
				drModification[TABLE.RowVersion] = (byte[]) this._rowVersions[enumerator.Key];
				drModification[TABLE.LastUpdateFunctionId] = (int) lastUpdateFunctionId;
				dtModifications.Rows.Add(drModification);
			}

			int uselessVariable = new StoredProcedure().TransactionalGetValue(STOREDPROC.UpdateProductModifications.StoredProcName, dtModifications);

			this.LoadProductModifications(divisionCode);
			return true;
		}




		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/17/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6332"> 
		///		<ExpectedInput>Active ProductModification Code</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6333"> 
		///		<ExpectedInput>Inactive ProductModificationCode</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns if the Modification Type is active for the division.  BR_26.32
		/// </summary>
		/// <param name="modificationCode"></param>
		/// <returns></returns>
		public bool GetModificationStatus(Common.ProductModification modificationCode)
		{
			char modChar = Common.Utility.GetProductModificationCharFromEnum(modificationCode);

			return ((bool)this._productModifications[modChar]);
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/20/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2133"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="603"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Unused - not applicable to this class.. Although this class doesn't really need to exist either
		/// </summary>
		/// <param name="dtRow"></param>
		/// <returns></returns>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			return dtRow;
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/20/2003</CreationDate>
		/// <summary>
		/// Unused
		/// </summary>
		/// <param name="dtRow"></param>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{

		}

		#endregion
	}
}





